/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

static const char __idstring[] = "@(#)$Id: mx_lanai_command.c,v 1.13 2006/01/20 23:42:34 patrick Exp $";
#include "mx_arch.h"
#include "mx_instance.h"
#include "mx_malloc.h"
#include "mx_misc.h"
#include "mx_peer.h"
#include "mx_pio.h"
#include "mcp_config.h"
#include "mx_stbar.h"



static inline int
mx_mcp_status_to_something_sane(int mcp_status)
{
  int status;

  switch (mcp_status) {
  case MX_MCP_CMD_OK:
    status = 0;
    break;

  case MX_MCP_CMD_ERROR_RANGE:
    status = ERANGE;
    break;

  case MX_MCP_CMD_ERROR_BUSY:
    status = EBUSY;
    break;

  case MX_MCP_CMD_ERROR_CLOSED:
    status = EBADF;
    break;

  case MX_MCP_CMD_UNKNOWN:
  default:
    status = ENXIO;
    break;
  }

  return (status);
}

int
mx_submit_command(mx_instance_state_t *is, struct cmdq_entry *entry)
{
  unsigned long flags;

  flags = 0; /* useless initialization to pacify -Wunused on platforms
		where flags are not used */
		
  if (mx_is_dead(is))
    return EIO;

  /* We take the cmdq spinlock so as to protect ourselves against another
     thread sumbitting a command and racing for the slot.  We must
     hold the mutex across the entire command submission in order to
     prevent the other thread from submitting a command for index N+1
     while we are filling out the command at index N */

  mx_spin_lock_irqsave(&is->cmdq.spinlock, flags);
  is->cmdq.len++;
  /* queue the command */
  STAILQ_INSERT_TAIL(&is->cmdq.host_cmdq, entry, entries);
  /* try to feed it to the MCP */
  mx_transfer_commands(is);
  mx_spin_unlock_irqrestore(&is->cmdq.spinlock, flags);
  return 0;
}

/* move queued commands from the software queue to the hardware
   queue. The is->cmdq.spinlock is held around this function.  

   This is called when adding software queue items via
   mx_submit_command(), and when the interrupt handler processes
   flow interrupts (this freeing space in the mcp). */

void
mx_transfer_commands(mx_instance_state_t *is)
{
  struct cmdq_entry *entry;
  int slot;

  while (is->cmdq.submitted != (is->cmdq.completed + is->cmdq.max_index) 
	 && !STAILQ_EMPTY(&is->cmdq.host_cmdq)) {
    entry = STAILQ_FIRST(&is->cmdq.host_cmdq);

    /* transfer the command from the software to hardware queue */
    STAILQ_REMOVE_HEAD(&is->cmdq.host_cmdq, entries);
    STAILQ_INSERT_TAIL(&is->cmdq.mcp_cmdq, entry, entries);
    slot = is->cmdq.submitted & is->cmdq.max_index;
    entry->cmd_seqnum = is->cmdq.submitted;
    MX_PIO_WRITE(&(is->cmdq.mcp[slot].data0), htonl(entry->cmd_data0));
    MX_PIO_WRITE(&(is->cmdq.mcp[slot].data1), htonl(entry->cmd_data1));
    MX_PIO_WRITE(&(is->cmdq.mcp[slot].seqnum), htonl(entry->cmd_seqnum));
    MX_PIO_WRITE(&(is->cmdq.mcp[slot].index), htons(entry->cmd_index));
    MX_STBAR();
    MX_PIO_WRITE(&(is->cmdq.mcp[slot].type), entry->cmd);
    MX_STBAR();
    entry->slot = slot;
    is->cmdq.submitted++;
  }
}

int
mx_status_command(mx_instance_state_t *is, uint32_t mcp_status, uint32_t mcp_data)
{
  int status;

  /* if the command completed with an error, try to translate the MCP 
     error code into something that the driver understand */
  if ((int)mcp_status < 0) {
    status = mx_mcp_status_to_something_sane(mcp_data);
  } else {
    status = 0;
  }
  return (status);
}

int
mx_wait_command(mx_instance_state_t *is, struct cmdq_entry *entry)
{
  unsigned long flags;
  int status;
  struct cmdq_entry *np;

  flags = 0;

  status = mx_sleep(entry->sync, MX_COMMAND_WAIT, MX_SLEEP_NOINTR);

  if (status) {
    /* command timed out, remove it since we're waking up and our
       stack is going away */

    mx_spin_lock_irqsave(&is->cmdq.spinlock, flags);
    STAILQ_FOREACH(np, &is->cmdq.host_cmdq, entries) {
      if (entry == np) {
	STAILQ_REMOVE(&is->cmdq.host_cmdq, np, cmdq_entry, entries);
	is->cmdq.len--;
	MX_WARN(("mx%d: removed entry from host cmdq after timeout\n",
		 is->id));
	goto removed;
      }
    }
    STAILQ_FOREACH(np, &is->cmdq.mcp_cmdq, entries) {
      if (entry == np) {
	STAILQ_REMOVE(&is->cmdq.mcp_cmdq, np, cmdq_entry, entries);
	is->cmdq.len--;
	MX_WARN(("mx%d: removed entry from mcp cmdq after timeout\n",
		 is->id));
	goto removed;
      }
      MX_WARN(("mx%d: entry not found in mcp cmdq after timeout?\n",
	       is->id));
    }
  removed:
    mx_spin_unlock_irqrestore(&is->cmdq.spinlock, flags);
    if (mx_is_dead(is))
      return EIO;
    MX_WARN (("command %d timed out (%d, %d), marking instance %d dead\n", 
	      entry->cmd,  is->cmdq.submitted, is->cmdq.completed, is->id));

    mx_mark_board_dead(is, MX_DEAD_COMMAND_TIMEOUT, entry->cmd);
  }
  if (mx_is_dead(is))
    return EIO;

  status = mx_status_command(is, entry->mcp_status, entry->mcp_data);
  return status;
}

/* send a command to the lanai */
int
mx_lanai_command(mx_instance_state_t *is, uint32_t cmd, uint32_t cmd_index, 
		 uint32_t cmd_data0, uint32_t cmd_data1, uint32_t *mcp_data, 
		 mx_sync_t *sync)
{
  int status;
  struct cmdq_entry entry;

  entry.func = NULL;
  entry.sync = sync;
  entry.cmd_data0 = cmd_data0;
  entry.cmd_data1 = cmd_data1;
  entry.cmd_index = cmd_index;
  entry.cmd = cmd;

  status = mx_submit_command(is, &entry);
  if (!status) {
    status = mx_wait_command(is, &entry);
    *mcp_data = entry.mcp_data;
  }
  return status;
}

static void
mx_async_lanai_command_done(void *arg, struct cmdq_entry *entry)
{
  mx_instance_state_t *is = (mx_instance_state_t *)arg;
  if (!mx_is_dead(is))
    mx_status_command(is, entry->mcp_status, entry->mcp_data);
  mx_kfree(entry);
}

int
mx_async_lanai_command(mx_instance_state_t *is, uint32_t cmd, uint32_t cmd_index, 
		       uint32_t cmd_data0, uint32_t cmd_data1)
{
  int status;
  struct cmdq_entry *entry;

  entry = mx_kmalloc(sizeof(*entry), MX_WAITOK);
  if (entry == NULL)
    return ENOMEM;

  entry->func = mx_async_lanai_command_done;
  entry->sync = NULL;
  entry->cmd_data0 = cmd_data0;
  entry->cmd_data1 = cmd_data1;
  entry->cmd_index = cmd_index;
  entry->cmd = cmd;

  status = mx_submit_command(is, entry);
  return status;
}

void
mx_lanai_command_complete(mx_instance_state_t *is, uint32_t slot, uint32_t mcp_data, 
			  int mcp_status)
{
  struct cmdq_entry *entry;

  mx_spin_lock(&is->cmdq.spinlock);
  entry = STAILQ_FIRST(&is->cmdq.mcp_cmdq);
  if (entry == NULL) {
    MX_WARN(("mx%d: %s called on slot 0x%x with emtpy command queue!\n",
	     is->id, __FUNCTION__, slot));
    mx_spin_unlock(&is->cmdq.spinlock);
    return;
  }
  STAILQ_REMOVE_HEAD(&is->cmdq.mcp_cmdq, entries);
  is->cmdq.completed++;
  is->cmdq.len--;
  if (!STAILQ_EMPTY(&is->cmdq.host_cmdq))
    mx_transfer_commands(is);
  mx_spin_unlock(&is->cmdq.spinlock);
  /* Typically, when this assert fires, 2 threads are sleeping
     on the same sync.  This is due to a lanai command being
     submitted using a sync which is used for somthing else.  If
     this assert ever triggers without driver error, it means
     the mcp is handling commands out of order, and we need to
     look in the mcp_cmdq for the command which just completed */
  mx_assert(slot = 0xdead || slot == entry->slot);
  entry->mcp_status = mcp_status;
  entry->mcp_data = mcp_data;
  if (entry->sync) {
    mx_wake(entry->sync);
  } else if (entry->func) {
    entry->func(is, entry);
  }
}

